// C SourceFile
// Created 6/20/2010; 12:21:50 AM

#include <tigcclib.h>
#include "extgraph.h"

#include "BasicTypes.h"
#include "GrayUtils.h"
#include "FontUtils.h"
#include "TimeUtils.h"
#include "BasicUtils.h"
#include "pyramid.h"
#include "pyramid_gfx.h"
#include "card_gfx.h"


// Global Variables
bool gameOver;
bool exitGame;
bool resetGame;
bool node_selected;

uint4 autoQuitCount;

int2 deal_pile[MAX_NUM_CARDS + 1];
int2 save_pile[MAX_NUM_CARDS + 1];
int2 draw_pile[MAX_NUM_CARDS + 1];
int2 done_pile[MAX_NUM_CARDS + 1];

int2 deal_pile_index;
int2 save_pile_index;
int2 draw_pile_index;
int2 done_pile_index;

int2 node_select_index;
int2 node_cursor_index;
int2 node_cursor_index_saved;

int2 deal_pile_count;
int2 deal_pile_count_saved;

// Prototypes
void _main(void);
int2 DrawTitleScreen(void);
void PlayGame(void);
void CheckExitGame(void);
void CheckResetGame(void);
void CheckAdjustContrast(void);
bool CheckDisplayUpdate(void);
void CheckAutoQuit(uint4 value);
void CheckGameOver(void);
void CheckGetCard(void);
void CheckSelectCard(void);
void CheckReplayDeal(void);
void CheckNewDeal(void);
void CheckDirectionKeyPress(void);
void ResetAutoQuit(void);
void DrawGameOver(void);
void ShuffleCards(void);
void DealCards(void);
void DrawCard(void);
void MoveCursor(uint1 direction);
bool CheckSelection(int2 card1, int2 card2);
void SelectCard(void);
void DisplayCard(int2 value, int2 x, int2 y, bool selected);
void DisplayAllCards(void);
uint1 GetDirectionKeyPress(void);



// Main Function
void _main(void)
{
	bool done = false;
	int2 choice = 0;

	exitGame = false;

	// we are entering the game for the first time
	// so wait for the Enter key to be released.
	while (_keytest (RR_ENTER));

	// define the font we want to use
	SetUserFont(intvfont - 32 * 8, 8, 8);

	// turn on double buffered grayscale graphics
	GrayScaleOn();

	// clear both both screen buffers
	GrayFillScreen(COLOR_White);
	SwapPlaneBuffers();
	GrayFillScreen(COLOR_White);
	SwapPlaneBuffers();

	// seed random number generator
	randomize();

	// initialize the autoquit counter
	ResetAutoQuit();

	while(!done)
	{
		choice = DrawTitleScreen();

		if (exitGame)
		{
			done = true;
		}
		else if (choice == 1)
		{
			PlayGame();
		}
//		else if (choice == 2)
//		{
//			DrawAboutScreen();
//		}
	}

	GrayScaleOff();
}

int2 DrawTitleScreen(void)
{
	bool done = false;
	int2 choice = 0;

	resetGame = false;
	deal_pile_count = 2;

	GrayFillScreen(COLOR_White);

	GrayUserDrawStr(24,24, (char*)"David Randall", COLOR_Black, 0);
	GrayUserDrawStr(48,48, (char*)"presents", COLOR_Black, 0);
	GrayUserDrawStr( 8,72, (char*)"PYRAMID SOLITAIRE", COLOR_Black, 0);

	SwapPlaneBuffers();

	while (!done)
	{
		CheckExitGame();
		CheckResetGame();
		CheckAdjustContrast();
		CheckAutoQuit(TITLE_AUTOTIMER);

		if (_keytest (RR_2ND))
		{
			while (_keytest (RR_2ND));
			choice = 1;
			done = true;
		}

		if (exitGame)
		{
			done = true;
		}

		Sleep(50);
	}

	return choice;
}

void DrawSelectScreen(void)
{
	bool done = false;
	int2 curChoice = 1;

	GrayFillScreen(COLOR_White);
	SwapPlaneBuffers();
	GrayFillScreen(COLOR_White);
	SwapPlaneBuffers();

	ResetAutoQuit();

	while(!done)
	{
		CheckAutoQuit(TITLE_AUTOTIMER);
		CheckExitGame();
		CheckResetGame();
		CheckAdjustContrast();

		if (exitGame || resetGame)
		{
			done = true;
		}
		else if (_keytest (RR_LEFT))
		{
			// wait until key is released
			while (_keytest (RR_LEFT));
			ResetAutoQuit();
			curChoice--;
		}
		else if (_keytest (RR_RIGHT))
		{
			// wait until key is released
			while (_keytest (RR_RIGHT));
			ResetAutoQuit();
			curChoice++;
		}
		else if (_keytest (RR_2ND))
		{
			while (_keytest (RR_2ND));
			//deal_pile_count = curChoice;
			deal_pile_count_saved = curChoice;
			done = true;
		}

		if (curChoice > 9)
			curChoice = 1;
		else if (curChoice < 1)
			curChoice = 9;

		if (CheckDisplayUpdate())
		{
			GrayFillScreen(COLOR_White);

			GrayUserDrawStr( 0,15, (char*)"Select the number of", COLOR_Black, 0);
			GrayUserDrawStr( 0,25, (char*)"  times to rotate   ", COLOR_Black, 0);
			GrayUserDrawStr( 0,35, (char*)" through draw pile  ", COLOR_Black, 0);
			GrayUserDrawStr( 4,72, (char*)" 1 2 3 4 5 6 7 8 9 ", COLOR_Black, 0);
			
			DisplayCard(54, curChoice*16-8, 64, false);

			SwapPlaneBuffers();
		}
		else
		{
			Sleep(50);
		}

	}
}

void PlayGame(void)
{
	bool done = false;

	gameOver = false;
	resetGame = false;

	DrawSelectScreen();

	GrayFillScreen(COLOR_White);
	SwapPlaneBuffers();
	GrayFillScreen(COLOR_White);
	SwapPlaneBuffers();

	ShuffleCards();
	DealCards();

	ResetAutoQuit();

	while(!done)
	{
		CheckAutoQuit(GAME_AUTOTIMER);
		CheckExitGame();
		CheckResetGame();
		CheckAdjustContrast();
		CheckReplayDeal();
		CheckNewDeal();

		if (exitGame || resetGame)
		{
			done = true;
		}
		else if (!gameOver)
		{
			CheckDirectionKeyPress();
			CheckGetCard();
			CheckSelectCard();
		}

		if (CheckDisplayUpdate())
		{
			GrayFillScreen(COLOR_White);
			DisplayAllCards();

			if (gameOver)
			{
				DrawGameOver();
			}

			SwapPlaneBuffers();
		}
		else
		{
			Sleep(50);
		}
	}
}

void ShuffleCards(void)
{
	int2 i = 0;
	int2 j = 0;
	int2 k = 0;
	int2 temp_card;

	// initialize the deck of cards
	for (i = 0; i < MAX_NUM_CARDS; i++)
	{
		// place the cards in order	from least to greatest
		deal_pile[i] = i+1;
	}

	deal_pile_index = 0;
	save_pile_index = 0;

	// shuffle the cards 3 times
	for (k = 0; k < 3; k++)
	{
		for (i = 0; i < MAX_NUM_CARDS; i++)
		{
			j = RandomNumber(MAX_NUM_CARDS);

			// swap the current card with the randomly selected card
			temp_card    = deal_pile[i];
			deal_pile[i] = deal_pile[j];
			deal_pile[j] = temp_card;
		}
	}

	// save this shuffled deck in case we want to try again
	memcpy(&save_pile[0], &deal_pile[0], sizeof(save_pile[0]) * MAX_NUM_CARDS);
}


void DealCards(void)
{
	int i;

	// initialize the draw and deal piles
	memset(&draw_pile[0], 0, sizeof(draw_pile[0]) * MAX_NUM_CARDS);
	memset(&done_pile[0], 0, sizeof(done_pile[0]) * MAX_NUM_CARDS);

	draw_pile_index = 0;
	done_pile_index = 0;

	// restore the deal deck from the saved deck
	memcpy(&deal_pile[0], &save_pile[0], sizeof(deal_pile[0]) * MAX_NUM_CARDS);

	deal_pile_index = MAX_NUM_CARDS - 1;

	for (i = 0; i < PYRAMID_NUM_CARDS; i++)
	{
		// move this card from the deal pile into the pyramid pile
		pyramid_node[i].value = deal_pile[51 - i];

		// mark this card as dealt by moving its pile's index
		deal_pile_index--;;
	}

	// initialize the node cursor index to the middle card on bottom row
	node_cursor_index = 24;
	node_cursor_index_saved = 24;

	// initialize the node select index to nothing
	node_select_index = INVALID_INDEX;
	node_selected = false;

	gameOver = false;
	
	deal_pile_count = deal_pile_count_saved;
}

void DrawCard(void)
{
	// Check if cards are still in the deal pile
	if (deal_pile_index >= 0)
	{
		// cards are still in the deal pile
		// so move one card from the deal pile to the draw pile
		draw_pile_index++;

		draw_pile[draw_pile_index] = deal_pile[deal_pile_index];
		deal_pile[deal_pile_index] = 0;

		deal_pile_index--;
	}
	else
	{
		// the deal pile is empty
		// move the entire draw pile back into the deal pile
		while (draw_pile_index > 0)
		{
			deal_pile_index++;

			deal_pile[deal_pile_index] = draw_pile[draw_pile_index];
			draw_pile[draw_pile_index] = 0;

			draw_pile_index--;
		}
		
		deal_pile_count--;
		
		if (deal_pile_count <= 0)
		{
			// exhausted the number of deal pile attempts
			gameOver = true;
	
			// disable the selected card feature
			node_selected = false;
		}

	}

	node_selected = false;
}

void MoveCursor(uint1 direction)
{
	int2 i;
	int2 open_node_list[7]; // max of 7 cards can be open at any one time
	int2 open_node_count = 0;
	int2 current_node_index;

	current_node_index = node_cursor_index;

	// find all open nodes
	for (i = 27; i >= 0; i--)
	{
		if (pyramid_node[i].value > INVALID_VALUE)
		{
			// this is a valid node, now check to see if it is open
			if ((pyramid_node[i].child1 < 0) ||
					((pyramid_node[i].child1 >= 0) &&
					 (pyramid_node[(int2)pyramid_node[i].child1].value <= INVALID_VALUE) &&
					 (pyramid_node[(int2)pyramid_node[i].child2].value <= INVALID_VALUE)))
			{
				// the node either has no children or the children have been removed
				// save this open node in the open node list
				open_node_list[open_node_count] = i;
				open_node_count++;

				// max of 7 cards can be open at any one time
				if (open_node_count >= 7)
					break;
			}
		}
	}


	if ((direction == DIR_Right) || (direction == DIR_Left))
	{
		// let's find the closest node from the open node list using Pythagorean's theorem
		float closest_distance = 100;
		int2 closest_node_index = INVALID_INDEX;

		// get the coordinates of the current node
		int2 node_x = pyramid_node[current_node_index].graph_x;
		int2 node_y = pyramid_node[current_node_index].graph_y;

		// check each node in the open list
		for (i = 0; i < open_node_count; i++)
		{
			// get the coordinates of this open node
			int2 open_node_index = open_node_list[i];
			int2 test_x = node_x - pyramid_node[open_node_index].graph_x;
			int2 test_y = node_y - pyramid_node[open_node_index].graph_y;

			// Pythagorean's theorem
			float distance = sqrt((float)((test_x * test_x) + (test_y * test_y)));

			// make the distance negative if we are attempting to move left
			if (direction == DIR_Left)
			{
				test_x *= (-1);
			}

			// check to see if this node is closest to the current node
			if ((distance < closest_distance) && (test_x <= 0) && (open_node_index != current_node_index))
			{
				// this node is now the closest node
				closest_distance = distance;
				closest_node_index = open_node_index;
			}
		}

		// did we find a closest open node?
		if (closest_node_index >= 0)
		{
			// a closer open node was found, so make it the current node
			node_cursor_index = closest_node_index;
			node_cursor_index_saved = node_cursor_index;
		}
	}
	else if (direction == DIR_Up)
	{
		// move the current node to the draw pile if it has cards in it
		if ((node_cursor_index != INVALID_INDEX) && (draw_pile_index > 0))
		{
			node_cursor_index_saved = node_cursor_index;
			node_cursor_index = INVALID_INDEX;
		}
	}
	else if (direction == DIR_Down)
	{
		// move the cursor back to the last saved position of the pyramid pile
		node_cursor_index = node_cursor_index_saved;
	}
}

bool CheckSelection(int2 card1, int2 card2)
{
	// check to see if these 2 cards add up to 13
	int2 value = card1 + card2;

	if ((value % 13) == 0)
		return true;
	else
		return false;
}

void SelectCard(void)
{
	bool move_cursor = false;

	// has a card already been selected?
	if (node_selected)
	{
		// a card has already been selected, now check the new card
		// first, make sure the currently selected card is not reselected
		if (node_cursor_index != node_select_index)
		{
			if ((node_cursor_index == INVALID_INDEX) && (node_select_index > INVALID_INDEX))
			{
				// the cursor is on a card from the draw pile and a second card has been selected from the pyramid
				if (CheckSelection(draw_pile[draw_pile_index], pyramid_node[node_select_index].value))
				{
					// these 2 cards add up to 13 so move them to the done pile
					done_pile_index++;
					done_pile[done_pile_index] = draw_pile[draw_pile_index];
					draw_pile[draw_pile_index] = 0;
					draw_pile_index--;

					pyramid_node[node_select_index].value = INVALID_VALUE;
					move_cursor = true;
				}

				// disable the selected card feature
				node_selected = false;
			}
			else if ((node_cursor_index > INVALID_INDEX) && (node_select_index == INVALID_INDEX))
			{
				// the cursor is on a card from the pyramid and a second card has been selected from the draw pile
				if (CheckSelection(draw_pile[draw_pile_index], pyramid_node[node_cursor_index].value))
				{
					// these 2 cards add up to 13 so move them to the done pile
					done_pile_index++;
					done_pile[done_pile_index] = draw_pile[draw_pile_index];
					draw_pile[draw_pile_index] = 0;
					draw_pile_index--;

					pyramid_node[node_cursor_index].value = INVALID_VALUE;
					move_cursor = true;
				}

				// disable the selected card feature
				node_selected = false;
			}
			else if ((node_cursor_index > INVALID_INDEX) && (node_select_index > INVALID_INDEX))
			{
				// the cursor is on a card from the pyramid and a second card has been selected from the pyramid
				if (CheckSelection(pyramid_node[node_cursor_index].value, pyramid_node[node_select_index].value))
				{
					// these 2 cards add up to 13 so move them to the done pile
					done_pile_index++;
					done_pile[done_pile_index] = pyramid_node[node_cursor_index].value;
					done_pile_index++;
					done_pile[done_pile_index] = pyramid_node[node_select_index].value;

					pyramid_node[node_cursor_index].value = INVALID_VALUE;
					pyramid_node[node_select_index].value = INVALID_VALUE;
					move_cursor = true;
				}

				// disable the selected card feature
				node_selected = false;
			}
		}
		else
		{
			// the current and selected cards are identical
			// disable the selected card feature
			node_selected = false;
		}
	}
	else
	{
		// the current card is the first card to be selected
		if (node_cursor_index == INVALID_INDEX)
		{
			// this card is from the draw pile
			if (CheckSelection(draw_pile[draw_pile_index], 0))
			{
				// this card is a King so move it to the done pile
				done_pile_index++;
				done_pile[done_pile_index] = draw_pile[draw_pile_index];
				draw_pile[draw_pile_index] = 0;
				draw_pile_index--;

				// disable the selected card feature
				node_selected = false;
			}
			else
			{
				// enable the selected card feature to indicate a card has been selected
				node_selected = true;
				node_select_index = node_cursor_index;
			}
		}
		else
		{
			// this card is from the pyramid
			if (CheckSelection(pyramid_node[node_cursor_index].value, 0))
			{
				// this card is a King so move it to the done pile
				done_pile_index++;
				done_pile[done_pile_index] = pyramid_node[node_cursor_index].value;

				pyramid_node[node_cursor_index].value = INVALID_VALUE;

				// disable the selected card feature
				node_selected = false;
				move_cursor = true;
			}
			else
			{
				// enable the selected card feature to indicate a card has been selected
				node_selected = true;
				node_select_index = node_cursor_index;
			}
		}
	}

	// move the cursor
	node_cursor_index = node_cursor_index_saved;
	MoveCursor(DIR_Right);
	MoveCursor(DIR_Left);
	node_cursor_index_saved = node_cursor_index;

	// check for game over
	if (pyramid_node[0].value == INVALID_VALUE)
	{
		// the final card in the pyramid has been moved to the done pile
		gameOver = true;

		// disable the selected card feature
		node_selected = false;
	}
}

void DisplayAllCards(void)
{
	int2 i;
	bool selected;
	char deal_pile_count_string[4];

	if (!gameOver)
	{
		// first let's draw the cards in the pyramid
		for (i = 0; i < PYRAMID_NUM_CARDS; i++)
		{
			if (pyramid_node[i].value > INVALID_VALUE)
			{
				if ((node_select_index == i) && (node_selected))
					selected = true;
				else
					selected = false;

				DisplayCard(pyramid_node[i].value, pyramid_node[i].x, pyramid_node[i].y, selected);
			}
		}
	}

	// now lets draw the cards in the deal pile
	if (deal_pile_index < 0)
		DisplayCard(0, 4, 4, false);
	else
		DisplayCard(53, 4, 4, false);
	
	sprintf(deal_pile_count_string, "%d", deal_pile_count);
	GrayUserDrawStr(8,32, deal_pile_count_string, COLOR_Black, 0);

	// now lets draw the cards in the draw pile
	if ((node_select_index == INVALID_INDEX) && (node_selected))
		selected = true;
	else
		selected = false;

	DisplayCard(draw_pile[draw_pile_index], 24, 4, selected);

	// now lets draw the cards in the done pile
	DisplayCard(done_pile[done_pile_index], 140, 4, false);

	// now lets draw the cursor box
	if (node_cursor_index == INVALID_INDEX)
	{
		DisplayCard(54, 24, 4, false);
	}
	else if (gameOver == false)
	{
		DisplayCard(54, pyramid_node[node_cursor_index].x, pyramid_node[node_cursor_index].y, false);
	}
}

void DisplayCard(int2 value, int2 x, int2 y, bool selected)
{
	uint2* pCardSprite   = (uint2*)card_sprites + (value * 72);
	uint2* pSelectSprite = (uint2*)card_sprites + (55 * 72);

	if (selected)
		GraySprite16_SMASK_R(x, y, 24, pSelectSprite, pCardSprite, pCardSprite + 48, PLANE_CurrentPlane0, PLANE_CurrentPlane1);
	else
		GraySprite16_SMASK_R(x, y, 24, pCardSprite, pCardSprite + 24, pCardSprite + 48, PLANE_CurrentPlane0, PLANE_CurrentPlane1);
}



void CheckExitGame(void)
{
	if (_keytest (RR_ESC))
	{
		// wait until key is released
		while (_keytest (RR_ESC));
		exitGame = true;
	}
}

void CheckResetGame(void)
{
	if (_keytest (RR_F3))
	{
		// wait until key is released
		while (_keytest (RR_F3));
		resetGame = true;
	}
}

void CheckAdjustContrast(void)
{
	if (_keytest (RR_MINUS))
	{
		// wait until key is released
		while (_keytest (RR_MINUS));
		OSContrastDn();
		ResetAutoQuit();
	}
	else if (_keytest (RR_PLUS))
	{
		// wait until key is released
		while (_keytest (RR_PLUS));
		OSContrastUp();
		ResetAutoQuit();
	}
}

bool CheckDisplayUpdate(void)
{
	bool key_was_pressed = false;

	if (autoQuitCount <= 1)
	{
		key_was_pressed = true;
	}

	return key_was_pressed;
}


void CheckAutoQuit(uint4 value)
{
	if (++autoQuitCount >= value)
	{
		exitGame = true;
	}
}

void ResetAutoQuit(void)
{
	autoQuitCount = 0;
}

void CheckDirectionKeyPress(void)
{
	uint1 direction = GetDirectionKeyPress();

	if (direction)
	{
		MoveCursor(direction);
		ResetAutoQuit();
	}
}

void CheckGetCard(void)
{
	if (_keytest (RR_DIAMOND))
	{
		// wait until key is released
		while (_keytest (RR_DIAMOND));
		DrawCard();
		ResetAutoQuit();
	}
}

void CheckSelectCard(void)
{
	if (_keytest (RR_2ND))
	{
		// wait until key is released
		while (_keytest (RR_2ND));
		SelectCard();
		ResetAutoQuit();
	}
}

void CheckReplayDeal(void)
{
	if (_keytest (RR_F1))
	{
		// wait until key is released
		while (_keytest (RR_F1));
		DealCards();
		ResetAutoQuit();
	}
}

void CheckNewDeal(void)
{
	if (_keytest (RR_F2))
	{
		// wait until key is released
		while (_keytest (RR_F2));
		ShuffleCards();
		DealCards();
		ResetAutoQuit();
	}
}

void DrawGameOver(void)
{
	if (deal_pile_count <= 0)
		GrayUserDrawStr(48,48, (char*)"YOU LOST", COLOR_Black, 0);
	else
		GrayUserDrawStr(48,48, (char*)"YOU WON!", COLOR_Black, 0);
}

uint1 GetDirectionKeyPress(void)
{
	static uint1 previous_direction = DIR_None;
	uint1 direction;

	if (_keytest (RR_UP))
	{
		direction = DIR_Up;
	}
	else if (_keytest (RR_DOWN))
	{
		direction = DIR_Down;
	}
	else if (_keytest (RR_LEFT))
	{
		direction = DIR_Left;
	}
	else if (_keytest (RR_RIGHT))
	{
		direction = DIR_Right;
	}
	else
	{
		direction = DIR_None;
	}

	// if the direction key is being held down, then simply return no direction key pressed
	if (direction == previous_direction)
		direction = DIR_None;
	else
		previous_direction = direction;

	return direction;
}
